#region Includes
using GroundhogLearning.GameKit.Utils;
using UnityEngine;
using UnityEngine.Events;
#endregion

namespace GroundhogLearning.GameKit
{
    [AddComponentMenu(EditorMenuConfig.CATEGORY_INTERACTIONS + "Generic Trigger")]
    public class GenericTrigger : MonoBehaviour
    {
        #region Variables

        [Header("Configuration")]
        [SerializeField, Tooltip("Set tag to detect only objects with the same tag")]
        private string _tag;

        [Header("Events")]
        [SerializeField, Tooltip("Event called when an object enters the trigger")]
        private UnityEvent OnColliderEntered;

        [SerializeField, Tooltip("Event called when an object stays in the trigger")]
        private UnityEvent OnColliderStay;

        [SerializeField, Tooltip("Event called when an object exits the trigger")]
        private UnityEvent OnColliderExited;

        #endregion

        private void Awake()
        {
            if (!TryGetComponent(out Collider collider))
            {
                Debug.LogWarning($"GenericTrigger | {gameObject.name} has no Collider.");
                return;
            }

            collider.isTrigger = true;
        }

        private void OnTriggerEnter(Collider other)
        {
            Debug.Log($"Generic Trigger| {other.name} entered {gameObject.name}");

            if (!IsValidCollider(other)) return;
            OnColliderEntered.Invoke();
        }
        private void OnTriggerStay(Collider other)
        {
            Debug.Log($"Generic Trigger| {other.name} stays in {gameObject.name}");

            if (!IsValidCollider(other)) return;
            OnColliderStay.Invoke();
        }
        private void OnTriggerExit(Collider other)
        {
            Debug.Log($"Generic Trigger| {other.name} exited {gameObject.name}");

            if (!IsValidCollider(other)) return;
            OnColliderExited.Invoke();
        }

        private bool IsValidCollider(Collider other)
        {
            return string.IsNullOrEmpty(_tag) || other.gameObject.CompareTag(_tag);
        }
    }
}